# -*- coding: utf-8 -*-
"""sklearnExample.ipynb

Kod wygenerowany automatycznie przez Colaboratory.

Plik oryginalny znajduje się pod adresem
    https://colab.research.google.com/drive/1V-FpqDogoEgsZLT7UiLgPNAhHJLfAqqP
"""

# <---- Kliknij "Run", aby zobaczyć dane wyjściowe.


print(__doc__)

# Autorzy: Vlad Niculae, Alexandre Gramfort
# Licencja: BSD 3 clause

import logging
from time import time

from numpy.random import RandomState
import matplotlib.pyplot as plt

from sklearn.datasets import fetch_olivetti_faces
from sklearn.cluster import MiniBatchKMeans
from sklearn import decomposition

# Informacje o postępie są przekazywane do standardowego wyjścia.
logging.basicConfig(level=logging.INFO,
                    format='%(asctime)s %(levelname)s %(message)s')
n_row, n_col = 2, 3
n_components = n_row * n_col
image_shape = (64, 64)
rng = RandomState(0)

# #############################################################################
# Wczytanie danych twarzy.
faces, _ = fetch_olivetti_faces(return_X_y=True, shuffle=True,
                                random_state=rng)
n_samples, n_features = faces.shape

# Wyśrodkowanie globalne.
faces_centered = faces - faces.mean(axis=0)

# Wyśrodkowanie lokalne.
faces_centered -= faces_centered.mean(axis=1).reshape(n_samples, -1)

print("Zbiór danych zawiera informacje o %d twarzach" % n_samples)


def plot_gallery(title, images, n_col=n_col, n_row=n_row, cmap=plt.cm.gray):
    plt.figure(figsize=(2. * n_col, 2.26 * n_row))
    plt.suptitle(title, size=16)
    for i, comp in enumerate(images):
        plt.subplot(n_row, n_col, i + 1)
        vmax = max(comp.max(), -comp.min())
        plt.imshow(comp.reshape(image_shape), cmap=cmap,
                   interpolation='nearest',
                   vmin=-vmax, vmax=vmax)
        plt.xticks(())
        plt.yticks(())
    plt.subplots_adjust(0.01, 0.05, 0.99, 0.93, 0.04, 0.)

# #############################################################################
# Lista różnych estymatorów przeznaczonych do przeanalizowania problemu i jego
# rozwiązania za pomocą API klastrowania.
estimators = [
    ('Eigenfaces - PCA using randomized SVD',
     decomposition.PCA(n_components=n_components, svd_solver='randomized',
                       whiten=True),
     True),

    ('Non-negative components - NMF',
     decomposition.NMF(n_components=n_components, init='nndsvda', tol=5e-3),
     False),

    ('Independent components - FastICA',
     decomposition.FastICA(n_components=n_components, whiten=True),
     True),

    ('Sparse comp. - MiniBatchSparsePCA',
     decomposition.MiniBatchSparsePCA(n_components=n_components, alpha=0.8,
                                      n_iter=100, batch_size=3,
                                      random_state=rng),
     True),

    ('MiniBatchDictionaryLearning',
        decomposition.MiniBatchDictionaryLearning(n_components=15, alpha=0.1,
                                                  n_iter=50, batch_size=3,
                                                  random_state=rng),
     True),

    ('Cluster centers - MiniBatchKMeans',
        MiniBatchKMeans(n_clusters=n_components, tol=1e-3, batch_size=20,
                        max_iter=50, random_state=rng),
     True),

    ('Factor Analysis components - FA',
     decomposition.FactorAnalysis(n_components=n_components, max_iter=20),
     True),
]


# #############################################################################
# Wyświetlenie próbki danych wejściowych.

plot_gallery("Pierwsze wyśrodkowane twarze ze zbioru Olivetti", faces_centered[:n_components])

# #############################################################################
# Estymacja i wyświetlenie wykresu.

for name, estimator, center in estimators:
    print("Wyodrębnianie najlepszych %d %s..." % (n_components, name))
    t0 = time()
    data = faces
    if center:
        data = faces_centered
    estimator.fit(data)
    train_time = (time() - t0)
    print("Zrobione w trakcie %0.3fs" % train_time)
    if hasattr(estimator, 'cluster_centers_'):
        components_ = estimator.cluster_centers_
    else:
        components_ = estimator.components_

    # Wyświetlenie obrazu przedstawiającego wariant dostarczony przez np.
    # atrybut noise_variance_ estymatora. Estymator Eigenfaces z użyciem
    # dekompozycji PCA zapewnia również skalarny wariant noise_variance_
    # (średnia wariantu pixelwise), który nie może być wyświetlony w postaci
    # obrazu więc go pomijamy.
    if (hasattr(estimator, 'noise_variance_') and
            estimator.noise_variance_.ndim > 0):  # Pominięcie przypadku Eigenfaces.
        plot_gallery("Pixelwise variance",
                     estimator.noise_variance_.reshape(1, -1), n_col=1,
                     n_row=1)
    plot_gallery('%s - czas uczenia %.1fs' % (name, train_time),
                 components_[:n_components])

plt.show()

# #############################################################################
# Różne ograniczenia zastosowane podczas uczenia słownikowego.
estimators = [
    ('Dictionary learning',
        decomposition.MiniBatchDictionaryLearning(n_components=15, alpha=0.1,
                                                  n_iter=50, batch_size=3,
                                                  random_state=rng),
     True),
    ('Dictionary learning - positive dictionary',
        decomposition.MiniBatchDictionaryLearning(n_components=15, alpha=0.1,
                                                  n_iter=50, batch_size=3,
                                                  random_state=rng,
                                                  positive_dict=True),
     True),
    ('Dictionary learning - positive code',
        decomposition.MiniBatchDictionaryLearning(n_components=15, alpha=0.1,
                                                  n_iter=50, batch_size=3,
                                                  fit_algorithm='cd',
                                                  random_state=rng,
                                                  positive_code=True),
     True),
    ('Dictionary learning - positive dictionary & code',
        decomposition.MiniBatchDictionaryLearning(n_components=15, alpha=0.1,
                                                  n_iter=50, batch_size=3,
                                                  fit_algorithm='cd',
                                                  random_state=rng,
                                                  positive_dict=True,
                                                  positive_code=True),
     True),
]


# #############################################################################
# Wyświetlenie próbki danych wejściowych.

plot_gallery("Pierwsze wyśrodkowane twarze ze zbioru Olivetti", faces_centered[:n_components],
             cmap=plt.cm.RdBu)

# #############################################################################
# Przeprowadzenie obliczeń i wyświetlenie wykresu.

for name, estimator, center in estimators:
    print("Wyodrębnianie najlepszych %d %s..." % (n_components, name))
    t0 = time()
    data = faces
    if center:
        data = faces_centered
    estimator.fit(data)
    train_time = (time() - t0)
    print("Zrobione w trakcie %0.3fs" % train_time)
    components_ = estimator.components_
    plot_gallery(name, components_[:n_components], cmap=plt.cm.RdBu)

plt.show()

"""Kod: https://scikit-learn.org/stable/auto_examples/decomposition/plot_faces_decomposition.html#sphx-glr-auto-examples-decomposition-plot-faces-decomposition-py

Finxter Email Academy (Free): https://blog.finxter.com/email-academy/
"""